<?php
/**
 * System Test and Debug Script
 * This script helps diagnose issues with your Supabase Table Editor installation
 */

// Start output buffering to capture any errors
ob_start();
$errors = [];
$warnings = [];
$info = [];

// Test 1: PHP Version
$info[] = "PHP Version: " . PHP_VERSION;
if (version_compare(PHP_VERSION, '7.4.0', '<')) {
    $errors[] = "PHP 7.4 or higher is required. Current version: " . PHP_VERSION;
}

// Test 2: Required Extensions
$requiredExtensions = ['curl', 'json', 'mbstring'];
foreach ($requiredExtensions as $ext) {
    if (extension_loaded($ext)) {
        $info[] = "✅ Extension '{$ext}' is loaded";
    } else {
        $errors[] = "❌ Required extension '{$ext}' is not loaded";
    }
}

// Test 3: File Permissions
$requiredFiles = [
    'config/config.php' => 'Configuration file',
    'includes/SupabaseClient.php' => 'Supabase client',
    'includes/TableManager.php' => 'Table manager',
    'includes/functions.php' => 'Utility functions',
    'assets/css/style.css' => 'Stylesheet',
    'assets/js/app.js' => 'JavaScript',
    'api/' => 'API directory'
];

foreach ($requiredFiles as $file => $description) {
    if (file_exists($file)) {
        if (is_readable($file)) {
            $info[] = "✅ {$description} exists and is readable";
        } else {
            $errors[] = "❌ {$description} exists but is not readable";
        }
    } else {
        $errors[] = "❌ {$description} does not exist: {$file}";
    }
}

// Test 4: Configuration
$configExists = file_exists('config/config.php');
if ($configExists) {
    require_once 'config/config.php';
    
    if (defined('SUPABASE_URL') && defined('SUPABASE_ANON_KEY') && defined('SUPABASE_SERVICE_KEY')) {
        $info[] = "✅ Configuration file loaded successfully";
        
        if (SUPABASE_URL === 'https://your-project-ref.supabase.co') {
            $warnings[] = "⚠️ Supabase URL not configured (using default value)";
        }
        
        if (SUPABASE_ANON_KEY === 'your-anon-key-here') {
            $warnings[] = "⚠️ Supabase anonymous key not configured";
        }
        
        if (SUPABASE_SERVICE_KEY === 'your-service-role-key-here') {
            $warnings[] = "⚠️ Supabase service role key not configured";
        }
        
        // Test 5: Supabase Connection
        if (SUPABASE_URL !== 'https://your-project-ref.supabase.co' && 
            SUPABASE_ANON_KEY !== 'your-anon-key-here') {
            
            try {
                require_once 'includes/SupabaseClient.php';
                $supabase = new SupabaseClient(SUPABASE_URL, SUPABASE_ANON_KEY, SUPABASE_SERVICE_KEY);
                
                // Test basic connection
                $tables = $supabase->getTables();
                $info[] = "✅ Successfully connected to Supabase";
                $info[] = "📊 Found " . count($tables) . " tables in database";
                
                if (count($tables) > 0) {
                    $info[] = "Tables: " . implode(', ', array_slice($tables, 0, 5)) . 
                             (count($tables) > 5 ? ' (and ' . (count($tables) - 5) . ' more)' : '');
                }
                
            } catch (Exception $e) {
                $errors[] = "❌ Failed to connect to Supabase: " . $e->getMessage();
            }
        }
        
    } else {
        $errors[] = "❌ Configuration file is missing required constants";
    }
} else {
    $warnings[] = "⚠️ Configuration file does not exist. Run setup.php first.";
}

// Test 6: API Endpoints
$apiFiles = [
    'api/get_tables.php' => 'Get tables endpoint',
    'api/get_table_data.php' => 'Get table data endpoint',
    'api/update_cell.php' => 'Update cell endpoint',
    'api/delete_row.php' => 'Delete row endpoint',
    'api/add_row.php' => 'Add row endpoint'
];

foreach ($apiFiles as $file => $description) {
    if (file_exists($file) && is_readable($file)) {
        $info[] = "✅ {$description} is available";
    } else {
        $errors[] = "❌ {$description} is missing or not readable";
    }
}

// Test 7: Directory Structure
$requiredDirs = [
    'config' => 'Configuration directory',
    'includes' => 'PHP includes directory',
    'assets' => 'Assets directory',
    'assets/css' => 'CSS directory',
    'assets/js' => 'JavaScript directory',
    'api' => 'API directory'
];

foreach ($requiredDirs as $dir => $description) {
    if (is_dir($dir)) {
        if (is_readable($dir)) {
            $info[] = "✅ {$description} exists and is accessible";
        } else {
            $warnings[] = "⚠️ {$description} exists but may not be accessible";
        }
    } else {
        $errors[] = "❌ {$description} does not exist";
    }
}

// Capture any output
$output = ob_get_clean();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>System Test - Supabase Table Editor</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .test-container {
            max-width: 1000px;
            margin: 30px auto;
            padding: 20px;
        }
        
        .test-section {
            background: white;
            margin: 20px 0;
            padding: 25px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }
        
        .test-results {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }
        
        .result-box {
            padding: 20px;
            border-radius: 8px;
            border-left: 4px solid;
        }
        
        .result-box.errors {
            background: #fff5f5;
            border-color: #e53e3e;
        }
        
        .result-box.warnings {
            background: #fffbeb;
            border-color: #f59e0b;
        }
        
        .result-box.info {
            background: #f0fff4;
            border-color: #38a169;
        }
        
        .result-box h3 {
            margin-top: 0;
            margin-bottom: 15px;
        }
        
        .result-box ul {
            margin: 0;
            padding-left: 20px;
        }
        
        .result-box li {
            margin-bottom: 8px;
            line-height: 1.4;
        }
        
        .status-summary {
            display: flex;
            justify-content: center;
            gap: 30px;
            margin: 30px 0;
            flex-wrap: wrap;
        }
        
        .status-item {
            text-align: center;
            padding: 15px 25px;
            border-radius: 50px;
            font-weight: bold;
            color: white;
            min-width: 120px;
        }
        
        .status-errors {
            background: #e53e3e;
        }
        
        .status-warnings {
            background: #f59e0b;
        }
        
        .status-info {
            background: #38a169;
        }
        
        .actions {
            text-align: center;
            margin: 30px 0;
        }
        
        .code-output {
            background: #1a202c;
            color: #e2e8f0;
            padding: 20px;
            border-radius: 8px;
            font-family: 'Courier New', monospace;
            font-size: 14px;
            white-space: pre-wrap;
            overflow-x: auto;
            margin: 20px 0;
        }
        
        .recommendations {
            background: #ebf8ff;
            border: 1px solid #90cdf4;
            border-radius: 8px;
            padding: 20px;
            margin: 20px 0;
        }
        
        .recommendations h4 {
            color: #2b6cb0;
            margin-top: 0;
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="container">
            <h1>🔧 System Test Results</h1>
            <p>Diagnostic report for your Supabase Table Editor installation</p>
        </div>
    </div>

    <div class="test-container">
        
        <!-- Status Summary -->
        <div class="test-section">
            <h2>📊 Summary</h2>
            <div class="status-summary">
                <div class="status-item status-errors">
                    <?php echo count($errors); ?> Errors
                </div>
                <div class="status-item status-warnings">
                    <?php echo count($warnings); ?> Warnings
                </div>
                <div class="status-item status-info">
                    <?php echo count($info); ?> Info
                </div>
            </div>
        </div>

        <!-- Test Results -->
        <div class="test-section">
            <h2>🧪 Test Results</h2>
            
            <div class="test-results">
                <?php if (!empty($errors)): ?>
                    <div class="result-box errors">
                        <h3>❌ Errors (<?php echo count($errors); ?>)</h3>
                        <ul>
                            <?php foreach ($errors as $error): ?>
                                <li><?php echo htmlspecialchars($error); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>

                <?php if (!empty($warnings)): ?>
                    <div class="result-box warnings">
                        <h3>⚠️ Warnings (<?php echo count($warnings); ?>)</h3>
                        <ul>
                            <?php foreach ($warnings as $warning): ?>
                                <li><?php echo htmlspecialchars($warning); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>

                <?php if (!empty($info)): ?>
                    <div class="result-box info">
                        <h3>ℹ️ Information (<?php echo count($info); ?>)</h3>
                        <ul>
                            <?php foreach ($info as $infoItem): ?>
                                <li><?php echo htmlspecialchars($infoItem); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Recommendations -->
        <?php if (!empty($errors) || !empty($warnings)): ?>
            <div class="test-section">
                <div class="recommendations">
                    <h4>💡 Recommendations</h4>
                    
                    <?php if (!empty($errors)): ?>
                        <h5>For Errors:</h5>
                        <ul>
                            <li>Check that all required files are present and have correct permissions</li>
                            <li>Ensure PHP extensions (curl, json, mbstring) are installed</li>
                            <li>Verify your Supabase credentials are correct</li>
                            <li>Run <code>setup.php</code> to configure the application</li>
                        </ul>
                    <?php endif; ?>
                    
                    <?php if (!empty($warnings)): ?>
                        <h5>For Warnings:</h5>
                        <ul>
                            <li>Complete the configuration by running <code>setup.php</code></li>
                            <li>Ensure all Supabase credentials are properly set</li>
                            <li>Check file and directory permissions</li>
                        </ul>
                    <?php endif; ?>
                </div>
            </div>
        <?php endif; ?>

        <!-- Actions -->
        <div class="test-section">
            <div class="actions">
                <h3>🚀 Next Steps</h3>
                
                <?php if (count($errors) === 0 && count($warnings) === 0): ?>
                    <p style="color: #38a169; font-weight: bold; font-size: 1.2em;">
                        🎉 All tests passed! Your installation is ready to use.
                    </p>
                    <a href="index.php" class="btn btn-success">Launch Table Editor</a>
                <?php elseif (count($errors) === 0): ?>
                    <p style="color: #f59e0b; font-weight: bold;">
                        ⚠️ Some warnings found, but the system should work.
                    </p>
                    <a href="setup.php" class="btn">Complete Setup</a>
                    <a href="index.php" class="btn btn-success">Try Anyway</a>
                <?php else: ?>
                    <p style="color: #e53e3e; font-weight: bold;">
                        ❌ Errors found. Please fix them before proceeding.
                    </p>
                    <a href="setup.php" class="btn">Run Setup</a>
                <?php endif; ?>
                
                <button onclick="location.reload()" class="btn btn-secondary">🔄 Run Tests Again</button>
            </div>
        </div>

        <!-- Technical Details -->
        <?php if (!empty($output)): ?>
            <div class="test-section">
                <h3>🔍 Technical Output</h3>
                <div class="code-output"><?php echo htmlspecialchars($output); ?></div>
            </div>
        <?php endif; ?>

        <!-- System Information -->
        <div class="test-section">
            <h3>💻 System Information</h3>
            <div class="code-output">
Operating System: <?php echo PHP_OS; ?>

PHP Version: <?php echo PHP_VERSION; ?>

Web Server: <?php echo $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown'; ?>

Document Root: <?php echo $_SERVER['DOCUMENT_ROOT'] ?? 'Unknown'; ?>

Current Directory: <?php echo __DIR__; ?>

Memory Limit: <?php echo ini_get('memory_limit'); ?>

Max Execution Time: <?php echo ini_get('max_execution_time'); ?>s

Error Reporting: <?php echo error_reporting(); ?>

Display Errors: <?php echo ini_get('display_errors') ? 'On' : 'Off'; ?>
            </div>
        </div>
    </div>
</body>
</html>
