<?php
/**
 * Installation and Setup Script for Supabase Table Editor
 * 
 * This script helps you set up the Supabase Table Editor
 * Run this file in your browser to complete the setup process
 */

// Check if already configured
$configExists = file_exists(__DIR__ . '/config/config.php');
$configComplete = false;

if ($configExists) {
    require_once 'config/config.php';
    $configComplete = defined('SUPABASE_URL') && 
                     defined('SUPABASE_ANON_KEY') && 
                     defined('SUPABASE_SERVICE_KEY') &&
                     SUPABASE_URL !== 'https://your-project-ref.supabase.co' &&
                     SUPABASE_ANON_KEY !== 'your-anon-key-here';
}

// Handle form submission
if ($_POST && isset($_POST['setup'])) {
    $supabaseUrl = trim($_POST['supabase_url']);
    $anonKey = trim($_POST['anon_key']);
    $serviceKey = trim($_POST['service_key']);
    
    // Validate inputs
    $errors = [];
    
    if (empty($supabaseUrl) || !filter_var($supabaseUrl, FILTER_VALIDATE_URL)) {
        $errors[] = "Please enter a valid Supabase URL";
    }
    
    if (empty($anonKey)) {
        $errors[] = "Please enter your Supabase anonymous key";
    }
    
    if (empty($serviceKey)) {
        $errors[] = "Please enter your Supabase service role key";
    }
    
    if (empty($errors)) {
        // Create config file
        $configContent = "<?php\n";
        $configContent .= "// Supabase Configuration\n";
        $configContent .= "// Auto-generated by setup script\n\n";
        $configContent .= "define('SUPABASE_URL', '" . addslashes($supabaseUrl) . "');\n";
        $configContent .= "define('SUPABASE_ANON_KEY', '" . addslashes($anonKey) . "');\n";
        $configContent .= "define('SUPABASE_SERVICE_KEY', '" . addslashes($serviceKey) . "');\n\n";
        $configContent .= "// Application Settings\n";
        $configContent .= "define('APP_NAME', 'Supabase Table Editor');\n";
        $configContent .= "define('DEFAULT_LIMIT', 50); // Default number of rows to fetch\n";
        $configContent .= "define('MAX_LIMIT', 1000); // Maximum number of rows allowed\n\n";
        $configContent .= "// Error reporting (set to 0 for production)\n";
        $configContent .= "error_reporting(E_ALL);\n";
        $configContent .= "ini_set('display_errors', 1);\n";
        $configContent .= "?>";
        
        if (file_put_contents(__DIR__ . '/config/config.php', $configContent)) {
            $success = "Configuration saved successfully! You can now use the Supabase Table Editor.";
            $configComplete = true;
        } else {
            $errors[] = "Failed to write configuration file. Please check file permissions.";
        }
    }
}

// Test connection if configured
$connectionStatus = null;
if ($configComplete) {
    try {
        require_once 'includes/SupabaseClient.php';
        $supabase = new SupabaseClient(SUPABASE_URL, SUPABASE_ANON_KEY, SUPABASE_SERVICE_KEY);
        $tables = $supabase->getTables();
        $connectionStatus = "✅ Connection successful! Found " . count($tables) . " tables.";
    } catch (Exception $e) {
        $connectionStatus = "❌ Connection failed: " . $e->getMessage();
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Supabase Table Editor - Setup</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .setup-container {
            max-width: 800px;
            margin: 50px auto;
            padding: 30px;
            background: white;
            border-radius: 10px;
            box-shadow: 0 5px 20px rgba(0,0,0,0.1);
        }
        
        .setup-header {
            text-align: center;
            margin-bottom: 40px;
        }
        
        .setup-header h1 {
            color: #667eea;
            margin-bottom: 10px;
        }
        
        .step {
            margin-bottom: 30px;
            padding: 20px;
            border: 1px solid #e9ecef;
            border-radius: 8px;
        }
        
        .step-number {
            display: inline-block;
            width: 30px;
            height: 30px;
            background: #667eea;
            color: white;
            border-radius: 50%;
            text-align: center;
            line-height: 30px;
            font-weight: bold;
            margin-right: 15px;
        }
        
        .step-title {
            display: inline-block;
            font-size: 1.2em;
            font-weight: bold;
            color: #495057;
        }
        
        .connection-status {
            padding: 15px;
            border-radius: 8px;
            margin: 20px 0;
            font-weight: bold;
        }
        
        .connection-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .connection-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .code-block {
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 4px;
            padding: 15px;
            font-family: 'Courier New', monospace;
            font-size: 14px;
            margin: 10px 0;
            overflow-x: auto;
        }
        
        .success-message {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
            padding: 15px;
            border-radius: 8px;
            margin: 20px 0;
        }
        
        .complete-section {
            text-align: center;
            padding: 30px;
            background: #f8f9fa;
            border-radius: 8px;
            margin: 30px 0;
        }
        
        .complete-section h2 {
            color: #28a745;
            margin-bottom: 15px;
        }
        
        .action-buttons {
            display: flex;
            gap: 15px;
            justify-content: center;
            flex-wrap: wrap;
        }
    </style>
</head>
<body>
    <div class="setup-container">
        <div class="setup-header">
            <h1>🚀 Supabase Table Editor Setup</h1>
            <p>Welcome! Let's get your Supabase Table Editor configured and ready to use.</p>
        </div>

        <?php if (isset($success)): ?>
            <div class="success-message">
                <?php echo htmlspecialchars($success); ?>
            </div>
        <?php endif; ?>

        <?php if (isset($errors) && !empty($errors)): ?>
            <div class="alert alert-danger">
                <strong>Setup Errors:</strong>
                <ul>
                    <?php foreach ($errors as $error): ?>
                        <li><?php echo htmlspecialchars($error); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <?php if ($configComplete): ?>
            <div class="complete-section">
                <h2>✅ Setup Complete!</h2>
                <p>Your Supabase Table Editor is configured and ready to use.</p>
                
                <?php if ($connectionStatus): ?>
                    <div class="connection-status <?php echo strpos($connectionStatus, '✅') === 0 ? 'connection-success' : 'connection-error'; ?>">
                        <?php echo htmlspecialchars($connectionStatus); ?>
                    </div>
                <?php endif; ?>
                
                <div class="action-buttons">
                    <a href="index.php" class="btn btn-success">🚀 Launch Table Editor</a>
                    <button onclick="location.reload()" class="btn">🔄 Test Connection Again</button>
                    <a href="?reconfigure=1" class="btn btn-secondary">⚙️ Reconfigure</a>
                </div>
            </div>
        <?php endif; ?>

        <?php if (!$configComplete || isset($_GET['reconfigure'])): ?>
            <div class="step">
                <span class="step-number">1</span>
                <span class="step-title">Get Your Supabase Credentials</span>
                <div style="margin-top: 15px;">
                    <p>You'll need the following from your Supabase project dashboard:</p>
                    <ul>
                        <li><strong>Project URL:</strong> Found in Settings → API</li>
                        <li><strong>Anon Key:</strong> Found in Settings → API</li>
                        <li><strong>Service Role Key:</strong> Found in Settings → API (keep this secret!)</li>
                    </ul>
                    <p><strong>How to find these:</strong></p>
                    <ol>
                        <li>Go to your <a href="https://supabase.com/dashboard" target="_blank">Supabase Dashboard</a></li>
                        <li>Select your project</li>
                        <li>Navigate to Settings → API</li>
                        <li>Copy the Project URL and API keys</li>
                    </ol>
                </div>
            </div>

            <div class="step">
                <span class="step-number">2</span>
                <span class="step-title">Configure Your Connection</span>
                <div style="margin-top: 15px;">
                    <form method="POST">
                        <div class="form-group">
                            <label for="supabase_url">Supabase Project URL:</label>
                            <input type="url" 
                                   id="supabase_url" 
                                   name="supabase_url" 
                                   class="form-control" 
                                   placeholder="https://your-project.supabase.co"
                                   value="<?php echo isset($_POST['supabase_url']) ? htmlspecialchars($_POST['supabase_url']) : ''; ?>"
                                   required>
                        </div>

                        <div class="form-group">
                            <label for="anon_key">Anonymous Key:</label>
                            <input type="text" 
                                   id="anon_key" 
                                   name="anon_key" 
                                   class="form-control" 
                                   placeholder="eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9..."
                                   value="<?php echo isset($_POST['anon_key']) ? htmlspecialchars($_POST['anon_key']) : ''; ?>"
                                   required>
                        </div>

                        <div class="form-group">
                            <label for="service_key">Service Role Key:</label>
                            <input type="password" 
                                   id="service_key" 
                                   name="service_key" 
                                   class="form-control" 
                                   placeholder="eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9..."
                                   value="<?php echo isset($_POST['service_key']) ? htmlspecialchars($_POST['service_key']) : ''; ?>"
                                   required>
                            <small style="color: #6c757d;">⚠️ Keep this key secure! It has full access to your database.</small>
                        </div>

                        <button type="submit" name="setup" class="btn btn-success">💾 Save Configuration</button>
                    </form>
                </div>
            </div>
        <?php endif; ?>

        <div class="step">
            <span class="step-number">3</span>
            <span class="step-title">Security Recommendations</span>
            <div style="margin-top: 15px;">
                <ul>
                    <li>🔒 Never commit your <code>config/config.php</code> file to version control</li>
                    <li>🚫 Restrict access to your service role key</li>
                    <li>🌐 Consider using environment variables for production</li>
                    <li>🔐 Set up Row Level Security (RLS) in Supabase for additional protection</li>
                    <li>👥 Only give access to trusted users</li>
                </ul>
            </div>
        </div>

        <div class="step">
            <span class="step-number">4</span>
            <span class="step-title">Features Overview</span>
            <div style="margin-top: 15px;">
                <p>Once configured, you'll be able to:</p>
                <ul>
                    <li>📊 <strong>View Tables:</strong> Browse all your Supabase tables</li>
                    <li>✏️ <strong>Edit Cells:</strong> Click any cell to edit its value inline</li>
                    <li>🔍 <strong>Filter Data:</strong> Use column filters to find specific records</li>
                    <li>🗑️ <strong>Delete Rows:</strong> Remove unwanted records with confirmation</li>
                    <li>➕ <strong>Add Rows:</strong> Insert new records using a simple form</li>
                    <li>📱 <strong>Responsive Design:</strong> Works on desktop and mobile devices</li>
                </ul>
            </div>
        </div>

        <div style="text-align: center; margin-top: 40px; padding-top: 20px; border-top: 1px solid #e9ecef;">
            <p style="color: #6c757d;">
                <strong>Need Help?</strong> Check the <code>README.md</code> file for detailed documentation.
            </p>
        </div>
    </div>

    <script>
        // Show/hide service key
        document.addEventListener('DOMContentLoaded', function() {
            const serviceKeyInput = document.getElementById('service_key');
            if (serviceKeyInput) {
                serviceKeyInput.addEventListener('focus', function() {
                    this.type = 'text';
                });
                serviceKeyInput.addEventListener('blur', function() {
                    this.type = 'password';
                });
            }
        });
    </script>
</body>
</html>
