class SupabaseTableEditor {
    constructor() {
        this.currentTable = null;
        this.currentFilters = {};
        this.currentPage = 1;
        this.editingCell = null;
        
        this.init();
    }
    
    init() {
        this.loadTables();
        this.bindEvents();
    }
    
    bindEvents() {
        // Table selection
        document.getElementById('tableSelect')?.addEventListener('change', (e) => {
            this.selectTable(e.target.value);
        });
        
        // Filter form
        document.getElementById('filterForm')?.addEventListener('submit', (e) => {
            e.preventDefault();
            this.applyFilters();
        });
        
        // Clear filters
        document.getElementById('clearFilters')?.addEventListener('click', () => {
            this.clearFilters();
        });
        
        // Pagination
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('page-btn')) {
                e.preventDefault();
                const page = parseInt(e.target.dataset.page);
                this.loadPage(page);
            }
        });
        
        // Cell editing
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('editable-cell')) {
                this.startEditing(e.target);
            }
        });
        
        // Delete row
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('delete-btn')) {
                e.preventDefault();
                const rowId = e.target.dataset.rowId;
                this.deleteRow(rowId);
            }
        });
        
        // Modal events
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('modal') || e.target.classList.contains('close')) {
                this.closeModal();
            }
        });
        
        // Search functionality
        document.getElementById('searchBtn')?.addEventListener('click', () => {
            const query = document.getElementById('searchInput')?.value;
            if (query) {
                this.searchTable(query);
            }
        });
        
        document.getElementById('searchInput')?.addEventListener('keypress', (e) => {
            if (e.key === 'Enter') {
                const query = e.target.value;
                if (query) {
                    this.searchTable(query);
                }
            }
        });
        
        // Export buttons
        document.getElementById('exportCsvBtn')?.addEventListener('click', () => {
            this.exportTable('csv');
        });
        
        document.getElementById('exportJsonBtn')?.addEventListener('click', () => {
            this.exportTable('json');
        });
        
        // Table statistics
        document.getElementById('tableStatsBtn')?.addEventListener('click', () => {
            this.showTableStats();
        });
        
        // Bulk delete
        document.getElementById('bulkDeleteBtn')?.addEventListener('click', () => {
            this.bulkDeleteRows();
        });
        
        // Master checkbox for row selection
        document.addEventListener('change', (e) => {
            if (e.target.id === 'masterCheckbox') {
                this.toggleAllRows();
            }
            if (e.target.classList.contains('row-checkbox')) {
                this.updateBulkActions();
            }
        });
        
        // Keyboard events
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape') {
                if (this.editingCell) {
                    this.cancelEditing();
                }
                this.closeModal();
            }
            
            if (e.key === 'Enter' && this.editingCell) {
                this.saveCell();
            }
        });
    }
    
    async loadTables() {
        try {
            this.showLoading('Loading tables...');
            
            const response = await fetch('api/get_tables.php');
            const data = await response.json();
            
            if (data.error) {
                throw new Error(data.error);
            }
            
            this.populateTableSelect(data.tables);
            this.hideLoading();
            
        } catch (error) {
            this.showError('Failed to load tables: ' + error.message);
            this.hideLoading();
        }
    }
    
    populateTableSelect(tables) {
        const select = document.getElementById('tableSelect');
        if (!select) return;
        
        select.innerHTML = '<option value="">Select a table...</option>';
        
        tables.forEach(table => {
            const option = document.createElement('option');
            option.value = table;
            option.textContent = this.formatTableName(table);
            select.appendChild(option);
        });
    }
    
    async selectTable(tableName) {
        if (!tableName) {
            this.clearTableData();
            return;
        }
        
        this.currentTable = tableName;
        this.currentPage = 1;
        this.currentFilters = {};
        
        // Update table title
        const tableTitle = document.getElementById('tableTitle');
        if (tableTitle) {
            tableTitle.textContent = this.formatTableName(tableName);
        }
        
        await this.loadTableData();
        this.setupFilters();
    }
    
    async loadTableData() {
        if (!this.currentTable) return;
        
        try {
            this.showLoading('Loading table data...');
            
            const params = new URLSearchParams({
                table: this.currentTable,
                page: this.currentPage,
                ...this.currentFilters
            });
            
            const response = await fetch(`api/get_table_data.php?${params}`);
            const data = await response.json();
            
            if (data.error) {
                throw new Error(data.error);
            }
            
            this.renderTable(data.data, data.columns);
            this.renderPagination(data.pagination);
            this.updateTableInfo(data.pagination.total_rows);
            
            this.hideLoading();
            
        } catch (error) {
            this.showError('Failed to load table data: ' + error.message);
            this.hideLoading();
        }
    }
    
    renderTable(data, columns) {
        const container = document.getElementById('tableData');
        if (!container) return;
        
        if (!data || data.length === 0) {
            container.innerHTML = '<div class="loading">No data found</div>';
            return;
        }
        
        let html = `
            <table class="data-table">
                <thead>
                    <tr>
                        <th class="checkbox-cell">
                            <input type="checkbox" id="masterCheckbox" title="Select all rows">
                        </th>
                        ${columns.map(col => `<th>${this.formatColumnName(col)}</th>`).join('')}
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
        `;
        
        data.forEach(row => {
            const rowId = row.id || row[Object.keys(row)[0]]; // Use id or first column as row identifier
            html += '<tr>';
            
            // Add checkbox cell
            html += `
                <td class="checkbox-cell">
                    <input type="checkbox" class="row-checkbox" value="${rowId}">
                </td>
            `;
            
            columns.forEach(column => {
                const value = row[column];
                const cellClass = this.getCellClass(value);
                const displayValue = this.formatCellValue(value);
                
                html += `
                    <td>
                        <div class="editable-cell ${cellClass}" 
                             data-row-id="${rowId}" 
                             data-column="${column}"
                             data-value="${this.escapeHtml(value)}"
                             title="Click to edit">
                            ${displayValue}
                        </div>
                    </td>
                `;
            });
            
            html += `
                <td class="actions-cell">
                    <button class="action-btn delete delete-btn" 
                            data-row-id="${rowId}"
                            title="Delete row">
                        Delete
                    </button>
                </td>
            `;
            
            html += '</tr>';
        });
        
        html += '</tbody></table>';
        container.innerHTML = html;
    }
    
    setupFilters() {
        if (!this.currentTable) return;
        
        // Get table columns for filter setup
        this.loadTableColumns().then(columns => {
            this.createFilterInputs(columns);
            this.showFiltersContainer();
            this.showAddRowButton();
            this.showEnhancedControls();
        });
    }
    
    async loadTableColumns() {
        try {
            const response = await fetch(`api/get_table_data.php?table=${this.currentTable}&limit=1`);
            const data = await response.json();
            
            if (data.error) {
                throw new Error(data.error);
            }
            
            return data.columns || [];
        } catch (error) {
            console.error('Failed to load table columns:', error);
            return [];
        }
    }
    
    createFilterInputs(columns) {
        const filterRow = document.getElementById('filterRow');
        if (!filterRow) return;
        
        filterRow.innerHTML = '';
        
        columns.forEach(column => {
            const div = document.createElement('div');
            div.className = 'form-group';
            div.innerHTML = `
                <label for="filter_${column}">${this.formatColumnName(column)}:</label>
                <input type="text" 
                       id="filter_${column}" 
                       name="${column}" 
                       class="form-control" 
                       placeholder="Filter by ${this.formatColumnName(column)}">
            `;
            filterRow.appendChild(div);
        });
    }
    
    showFiltersContainer() {
        const container = document.getElementById('filtersContainer');
        if (container) {
            container.style.display = 'block';
        }
    }
    
    hideFiltersContainer() {
        const container = document.getElementById('filtersContainer');
        if (container) {
            container.style.display = 'none';
        }
    }
    
    showEnhancedControls() {
        const controls = document.getElementById('enhancedControls');
        if (controls) {
            controls.style.display = 'block';
        }
    }
    
    hideEnhancedControls() {
        const controls = document.getElementById('enhancedControls');
        if (controls) {
            controls.style.display = 'none';
        }
    }
    
    showAddRowButton() {
        const button = document.getElementById('addRowBtn');
        if (button) {
            button.style.display = 'inline-block';
        }
    }
    
    hideAddRowButton() {
        const button = document.getElementById('addRowBtn');
        if (button) {
            button.style.display = 'none';
        }
    }
    
    applyFilters() {
        const form = document.getElementById('filterForm');
        if (!form) return;
        
        const formData = new FormData(form);
        this.currentFilters = {};
        
        for (let [key, value] of formData.entries()) {
            if (value.trim()) {
                this.currentFilters[key] = value.trim();
            }
        }
        
        this.currentPage = 1;
        this.loadTableData();
    }
    
    clearFilters() {
        this.currentFilters = {};
        this.currentPage = 1;
        
        // Clear form inputs
        const form = document.getElementById('filterForm');
        if (form) {
            form.reset();
        }
        
        this.loadTableData();
    }
    
    loadPage(page) {
        this.currentPage = page;
        this.loadTableData();
    }
    
    startEditing(cell) {
        if (this.editingCell) {
            this.cancelEditing();
        }
        
        this.editingCell = cell;
        const currentValue = cell.dataset.value || '';
        const cellType = this.detectCellType(currentValue);
        
        cell.classList.add('editing');
        
        // Create appropriate input based on data type
        let inputHtml = '';
        
        if (cellType === 'boolean') {
            inputHtml = `
                <select class="cell-input">
                    <option value="true" ${currentValue === 'true' ? 'selected' : ''}>true</option>
                    <option value="false" ${currentValue === 'false' ? 'selected' : ''}>false</option>
                    <option value="" ${currentValue === '' || currentValue === 'null' ? 'selected' : ''}>NULL</option>
                </select>
            `;
        } else if (cellType === 'json') {
            inputHtml = `<textarea class="cell-input" rows="3">${this.escapeHtml(currentValue)}</textarea>`;
        } else {
            const inputType = cellType === 'number' ? 'number' : 'text';
            inputHtml = `<input type="${inputType}" class="cell-input" value="${this.escapeHtml(currentValue)}" />`;
        }
        
        cell.innerHTML = inputHtml;
        
        const input = cell.querySelector('.cell-input');
        input.focus();
        
        if (input.tagName === 'INPUT' || input.tagName === 'TEXTAREA') {
            input.select();
        }
        
        // Handle input events
        input.addEventListener('blur', () => this.saveCell());
        input.addEventListener('keydown', (e) => {
            e.stopPropagation();
            if (e.key === 'Tab') {
                e.preventDefault();
                this.saveCell();
                this.focusNextCell(cell);
            }
        });
    }
    
    detectCellType(value) {
        if (value === 'true' || value === 'false') return 'boolean';
        if (!isNaN(value) && value !== '') return 'number';
        if (this.isJsonString(value)) return 'json';
        return 'text';
    }
    
    isJsonString(str) {
        try {
            JSON.parse(str);
            return true;
        } catch (e) {
            return false;
        }
    }
    
    focusNextCell(currentCell) {
        const row = currentCell.closest('tr');
        const cells = Array.from(row.querySelectorAll('.editable-cell'));
        const currentIndex = cells.indexOf(currentCell);
        
        if (currentIndex < cells.length - 1) {
            // Focus next cell in same row
            this.startEditing(cells[currentIndex + 1]);
        } else {
            // Focus first cell in next row
            const nextRow = row.nextElementSibling;
            if (nextRow) {
                const firstCell = nextRow.querySelector('.editable-cell');
                if (firstCell) {
                    this.startEditing(firstCell);
                }
            }
        }
    }
    
    async saveCell() {
        if (!this.editingCell) return;
        
        const input = this.editingCell.querySelector('.cell-input');
        const newValue = input.value;
        const rowId = this.editingCell.dataset.rowId;
        const column = this.editingCell.dataset.column;
        const oldValue = this.editingCell.dataset.value;
        
        if (newValue === oldValue) {
            this.cancelEditing();
            return;
        }
        
        try {
            // Show loading state
            this.editingCell.classList.add('saving');
            
            const response = await fetch('api/update_cell.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    table: this.currentTable,
                    row_id: rowId,
                    column: column,
                    value: newValue
                })
            });
            
            const data = await response.json();
            
            if (data.error) {
                throw new Error(data.error);
            }
            
            // Update the cell
            this.editingCell.dataset.value = newValue;
            this.editingCell.classList.remove('editing', 'saving');
            this.editingCell.classList.add('cell-updated');
            this.editingCell.innerHTML = this.formatCellValue(newValue);
            
            // Remove success highlighting after 2 seconds
            setTimeout(() => {
                this.editingCell.classList.remove('cell-updated');
            }, 2000);
            
            this.showSuccess('Cell updated successfully');
            
        } catch (error) {
            this.editingCell.classList.remove('saving');
            this.editingCell.classList.add('cell-error');
            this.showError('Failed to update cell: ' + error.message);
            this.cancelEditing();
            
            // Remove error highlighting after 3 seconds
            setTimeout(() => {
                this.editingCell.classList.remove('cell-error');
            }, 3000);
        }
        
        this.editingCell = null;
    }
    
    cancelEditing() {
        if (!this.editingCell) return;
        
        const originalValue = this.editingCell.dataset.value;
        this.editingCell.classList.remove('editing');
        this.editingCell.innerHTML = this.formatCellValue(originalValue);
        this.editingCell = null;
    }
    
    async deleteRow(rowId) {
        if (!confirm('Are you sure you want to delete this row? This action cannot be undone.')) {
            return;
        }
        
        try {
            const response = await fetch('api/delete_row.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    table: this.currentTable,
                    row_id: rowId
                })
            });
            
            const data = await response.json();
            
            if (data.error) {
                throw new Error(data.error);
            }
            
            this.showSuccess('Row deleted successfully');
            this.loadTableData(); // Reload table data
            
        } catch (error) {
            this.showError('Failed to delete row: ' + error.message);
        }
    }
    
    renderPagination(pagination) {
        const container = document.getElementById('pagination');
        if (!container) return;
        
        const { current_page, total_pages, has_prev, has_next } = pagination;
        
        let html = `
            <div class="pagination-info">
                Page ${current_page} of ${total_pages}
            </div>
        `;
        
        if (has_prev) {
            html += `<button class="btn page-btn" data-page="${current_page - 1}">Previous</button>`;
        }
        
        // Show page numbers
        const startPage = Math.max(1, current_page - 2);
        const endPage = Math.min(total_pages, current_page + 2);
        
        for (let i = startPage; i <= endPage; i++) {
            const activeClass = i === current_page ? 'btn-active' : '';
            html += `<button class="btn page-btn ${activeClass}" data-page="${i}">${i}</button>`;
        }
        
        if (has_next) {
            html += `<button class="btn page-btn" data-page="${current_page + 1}">Next</button>`;
        }
        
        container.innerHTML = html;
    }
    
    updateTableInfo(totalRows) {
        const element = document.getElementById('tableInfo');
        if (element) {
            element.textContent = `${totalRows} rows`;
        }
    }
    
    clearTableData() {
        const container = document.getElementById('tableData');
        if (container) {
            container.innerHTML = '<div class="loading">Select a table to view data</div>';
        }
        
        const pagination = document.getElementById('pagination');
        if (pagination) {
            pagination.innerHTML = '';
        }
        
        const tableInfo = document.getElementById('tableInfo');
        if (tableInfo) {
            tableInfo.textContent = 'No data';
        }
        
        this.hideFiltersContainer();
        this.hideAddRowButton();
        this.hideEnhancedControls();
        
        const bulkActions = document.getElementById('bulkActions');
        if (bulkActions) {
            bulkActions.style.display = 'none';
        }
        
        const tableTitle = document.getElementById('tableTitle');
        if (tableTitle) {
            tableTitle.textContent = 'Select a Table';
        }
    }
    
    async showAddRowModal() {
        if (!this.currentTable) return;
        
        try {
            const columns = await this.loadTableColumns();
            
            let formHtml = '<form id="addRowForm">';
            
            columns.forEach(column => {
                // Skip auto-increment ID columns
                if (column.toLowerCase() === 'id') return;
                
                formHtml += `
                    <div class="form-group">
                        <label for="add_${column}">${this.formatColumnName(column)}:</label>
                        <input type="text" 
                               id="add_${column}" 
                               name="${column}" 
                               class="form-control" 
                               placeholder="Enter ${this.formatColumnName(column)}">
                    </div>
                `;
            });
            
            formHtml += `
                <div style="text-align: right; margin-top: 20px;">
                    <button type="button" class="btn btn-secondary" onclick="window.supabaseApp.closeModal()">Cancel</button>
                    <button type="submit" class="btn btn-success">Add Row</button>
                </div>
            </form>`;
            
            this.showModal('Add New Row', formHtml);
            
            // Handle form submission
            document.getElementById('addRowForm')?.addEventListener('submit', (e) => {
                e.preventDefault();
                this.submitAddRow();
            });
            
        } catch (error) {
            this.showError('Failed to show add row modal: ' + error.message);
        }
    }
    
    async submitAddRow() {
        const form = document.getElementById('addRowForm');
        if (!form) return;
        
        const formData = new FormData(form);
        const data = {};
        
        for (let [key, value] of formData.entries()) {
            if (value.trim()) {
                data[key] = value.trim();
            }
        }
        
        if (Object.keys(data).length === 0) {
            this.showError('Please enter at least one value');
            return;
        }
        
        try {
            const response = await fetch('api/add_row.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    table: this.currentTable,
                    data: data
                })
            });
            
            const result = await response.json();
            
            if (result.error) {
                throw new Error(result.error);
            }
            
            this.showSuccess('Row added successfully');
            this.closeModal();
            this.loadTableData(); // Reload table data
            
        } catch (error) {
            this.showError('Failed to add row: ' + error.message);
        }
    }
    
    // Utility methods
    formatTableName(name) {
        return name.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
    }
    
    formatColumnName(name) {
        return name.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
    }
    
    formatCellValue(value) {
        if (value === null || value === undefined) {
            return '<span class="cell-null">NULL</span>';
        }
        
        if (typeof value === 'boolean') {
            return `<span class="cell-boolean ${value}">${value ? 'true' : 'false'}</span>`;
        }
        
        if (typeof value === 'number') {
            return `<span class="cell-number">${value}</span>`;
        }
        
        if (typeof value === 'object') {
            return `<span class="cell-json">${JSON.stringify(value)}</span>`;
        }
        
        // Check if it's a date
        if (typeof value === 'string' && /^\d{4}-\d{2}-\d{2}/.test(value)) {
            return `<span class="cell-date">${value}</span>`;
        }
        
        return this.escapeHtml(String(value));
    }
    
    getCellClass(value) {
        if (value === null || value === undefined) return 'cell-null';
        if (typeof value === 'boolean') return 'cell-boolean';
        if (typeof value === 'number') return 'cell-number';
        if (typeof value === 'object') return 'cell-json';
        if (typeof value === 'string' && /^\d{4}-\d{2}-\d{2}/.test(value)) return 'cell-date';
        return '';
    }
    
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    showLoading(message = 'Loading...') {
        const container = document.getElementById('tableData');
        if (container) {
            container.innerHTML = `
                <div class="loading">
                    <div class="spinner"></div>
                    ${message}
                </div>
            `;
        }
    }
    
    hideLoading() {
        // Loading will be replaced by actual content
    }
    
    showError(message) {
        this.showAlert(message, 'danger');
    }
    
    showSuccess(message) {
        this.showAlert(message, 'success');
    }
    
    showAlert(message, type = 'info') {
        // Remove existing alerts
        const existing = document.querySelectorAll('.alert');
        existing.forEach(alert => alert.remove());
        
        const alert = document.createElement('div');
        alert.className = `alert alert-${type}`;
        alert.textContent = message;
        
        const container = document.querySelector('.container');
        if (container) {
            container.insertBefore(alert, container.firstChild);
            
            // Auto-remove after 5 seconds
            setTimeout(() => {
                alert.remove();
            }, 5000);
        }
    }
    
    showModal(title, content) {
        const modal = document.getElementById('modal');
        if (!modal) return;
        
        const titleElement = modal.querySelector('.modal-title');
        const bodyElement = modal.querySelector('.modal-body');
        
        if (titleElement) titleElement.textContent = title;
        if (bodyElement) bodyElement.innerHTML = content;
        
        modal.classList.add('show');
    }
    
    closeModal() {
        const modal = document.getElementById('modal');
        if (modal) {
            modal.classList.remove('show');
        }
    }
    
    // New enhanced features
    
    /**
     * Search functionality across all columns
     */
    async searchTable(query) {
        if (!this.currentTable || !query.trim()) {
            return;
        }
        
        try {
            this.showLoading('Searching...');
            
            const response = await fetch(`api/search_tables.php?table=${this.currentTable}&q=${encodeURIComponent(query)}`);
            const data = await response.json();
            
            if (data.error) {
                throw new Error(data.error);
            }
            
            this.renderSearchResults(data.results, data.query);
            this.hideLoading();
            
        } catch (error) {
            this.showError('Search failed: ' + error.message);
            this.hideLoading();
        }
    }
    
    /**
     * Render search results
     */
    renderSearchResults(results, query) {
        const tableData = document.getElementById('tableData');
        if (!tableData) return;
        
        if (results.length === 0) {
            tableData.innerHTML = `
                <div class="no-results">
                    <h3>No results found for "${this.escapeHtml(query)}"</h3>
                    <p>Try different search terms or check your spelling.</p>
                    <button onclick="location.reload()" class="btn">Clear Search</button>
                </div>
            `;
            return;
        }
        
        // Render results similar to regular table data
        const columns = Object.keys(results[0]);
        this.renderTable(results, columns);
        
        // Update table info
        const tableInfo = document.getElementById('tableInfo');
        if (tableInfo) {
            tableInfo.textContent = `${results.length} search results for "${query}"`;
        }
    }
    
    /**
     * Export table data
     */
    async exportTable(format = 'csv') {
        if (!this.currentTable) {
            this.showError('No table selected');
            return;
        }
        
        try {
            this.showLoading('Preparing export...');
            
            // Get all current table data (apply filters if any)
            const params = new URLSearchParams({
                table: this.currentTable,
                limit: 10000, // Export more data
                ...this.currentFilters
            });
            
            const response = await fetch(`api/get_table_data.php?${params}`);
            const tableData = await response.json();
            
            if (tableData.error) {
                throw new Error(tableData.error);
            }
            
            // Send export request
            const exportResponse = await fetch('api/export_table.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    table: this.currentTable,
                    data: tableData.data,
                    format: format
                })
            });
            
            const exportData = await exportResponse.json();
            
            if (exportData.error) {
                throw new Error(exportData.error);
            }
            
            // Trigger download
            window.location.href = exportData.download_url;
            
            this.showSuccess(`Exported ${exportData.records} records as ${format.toUpperCase()}`);
            this.hideLoading();
            
        } catch (error) {
            this.showError('Export failed: ' + error.message);
            this.hideLoading();
        }
    }
    
    /**
     * Bulk delete selected rows
     */
    async bulkDeleteRows() {
        const checkboxes = document.querySelectorAll('.row-checkbox:checked');
        const rowIds = Array.from(checkboxes).map(cb => cb.value);
        
        if (rowIds.length === 0) {
            this.showError('No rows selected for deletion');
            return;
        }
        
        if (!confirm(`Are you sure you want to delete ${rowIds.length} selected rows?`)) {
            return;
        }
        
        try {
            this.showLoading(`Deleting ${rowIds.length} rows...`);
            
            const response = await fetch('api/bulk_delete.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    table: this.currentTable,
                    rows: rowIds
                })
            });
            
            const data = await response.json();
            
            if (data.error) {
                throw new Error(data.error);
            }
            
            if (data.errors.length > 0) {
                this.showAlert(`Deleted ${data.deleted_count}/${data.total_requested} rows. Some errors occurred: ${data.errors.join(', ')}`, 'warning');
            } else {
                this.showSuccess(`Successfully deleted ${data.deleted_count} rows`);
            }
            
            // Reload table data
            await this.loadTableData();
            this.hideLoading();
            
        } catch (error) {
            this.showError('Bulk delete failed: ' + error.message);
            this.hideLoading();
        }
    }
    
    /**
     * Toggle all row checkboxes
     */
    toggleAllRows() {
        const masterCheckbox = document.getElementById('masterCheckbox');
        const checkboxes = document.querySelectorAll('.row-checkbox');
        
        checkboxes.forEach(checkbox => {
            checkbox.checked = masterCheckbox.checked;
        });
        
        this.updateBulkActions();
    }
    
    /**
     * Update bulk action buttons visibility
     */
    updateBulkActions() {
        const checkboxes = document.querySelectorAll('.row-checkbox:checked');
        const bulkActions = document.getElementById('bulkActions');
        
        if (bulkActions) {
            bulkActions.style.display = checkboxes.length > 0 ? 'block' : 'none';
        }
    }
    
    /**
     * Show table statistics modal
     */
    async showTableStats() {
        if (!this.currentTable) {
            this.showError('No table selected');
            return;
        }
        
        try {
            this.showLoading('Loading statistics...');
            
            const response = await fetch(`api/get_table_stats.php?table=${this.currentTable}`);
            const data = await response.json();
            
            if (data.error) {
                throw new Error(data.error);
            }
            
            const stats = data.stats;
            const content = `
                <h3>📊 Table Statistics: ${this.formatTableName(this.currentTable)}</h3>
                <div class="stats-grid">
                    <div class="stat-item">
                        <div class="stat-number">${stats.total_rows}</div>
                        <div class="stat-label">Total Rows</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number">${stats.total_columns}</div>
                        <div class="stat-label">Total Columns</div>
                    </div>
                </div>
                <div class="columns-list">
                    <h4>Columns:</h4>
                    <ul>
                        ${stats.columns.map(col => `<li><code>${col}</code></li>`).join('')}
                    </ul>
                </div>
            `;
            
            this.showModal('Table Statistics', content);
            this.hideLoading();
            
        } catch (error) {
            this.showError('Failed to load statistics: ' + error.message);
            this.hideLoading();
        }
    }
}

// Initialize the application when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.supabaseApp = new SupabaseTableEditor();
});
