<?php
header('Content-Type: application/json');
require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/SupabaseClient.php';
require_once '../includes/TableManager.php';

try {
    // Validate request method
    if (getRequestMethod() !== 'GET') {
        throw new Exception('Only GET method allowed');
    }
    
    // Get parameters
    $table = arrayGet($_GET, 'table');
    $page = max(1, intval(arrayGet($_GET, 'page', 1)));
    $limit = min(MAX_LIMIT, max(1, intval(arrayGet($_GET, 'limit', DEFAULT_LIMIT))));
    
    // Validate required parameters
    if (empty($table)) {
        throw new Exception('Table name is required');
    }
    
    // Initialize Supabase client
    $supabase = new SupabaseClient(SUPABASE_URL, SUPABASE_ANON_KEY, SUPABASE_SERVICE_KEY);
    $tableManager = new TableManager($supabase);
    
    // Sanitize table name
    $table = $tableManager->sanitizeTableName($table);
    
    // Get table columns
    $columns = $tableManager->getTableColumns($table);
    
    if (empty($columns)) {
        throw new Exception('Table not found or has no columns');
    }
    
    // Process filters
    $filters = [];
    foreach ($_GET as $key => $value) {
        if (!in_array($key, ['table', 'page', 'limit']) && !empty($value)) {
            $sanitizedKey = $tableManager->sanitizeColumnName($key);
            if (in_array($sanitizedKey, $columns)) {
                $filters[$sanitizedKey] = $value;
            }
        }
    }
    
    // Validate filters
    $validFilters = $tableManager->validateFilters($filters, $columns);
    
    // Get table data
    $data = $tableManager->getTableData($table, $validFilters, $page, $limit);
    
    // Get total count for pagination
    $totalRows = $tableManager->getRowCount($table, $validFilters);
    
    // Generate pagination info
    $pagination = generatePagination($page, $totalRows, $limit);
    
    successResponse([
        'data' => $data,
        'columns' => $columns,
        'pagination' => $pagination,
        'filters' => $validFilters,
        'table_name' => $table
    ]);
    
} catch (Exception $e) {
    logError("Error getting table data: " . $e->getMessage(), [
        'table' => arrayGet($_GET, 'table'),
        'page' => arrayGet($_GET, 'page'),
        'filters' => arrayGet($_GET, 'filters', [])
    ]);
    errorResponse($e->getMessage());
}
?>
